<?php

namespace App\Http\Services\Payment;

use PayPalCheckoutSdk\Core\PayPalHttpClient;
use PayPalCheckoutSdk\Core\SandboxEnvironment;
use PayPalCheckoutSdk\Core\LiveEnvironment;
use PayPalCheckoutSdk\Orders\OrdersCreateRequest;
use PayPalCheckoutSdk\Orders\OrdersCaptureRequest;
use Illuminate\Support\Facades\Session;

class PayPalService
{
    protected $client;

    public function __construct()
    {
        $clientId = get_payment_method('paypal_key');
        $clientSecret = get_payment_method('paypal_secret');
        $environment = get_payment_method('paypal_mode') == 1
            ? new SandboxEnvironment($clientId, $clientSecret)
            : new LiveEnvironment($clientId, $clientSecret);

        $this->client = new PayPalHttpClient($environment);
    }

    // Create PayPal order and redirect user
    public function createOrder($customerInfo)
    {
        $amount = $customerInfo['total_amount'] / (get_payment_method('stripe_conversion') ?? 1);

        $request = new OrdersCreateRequest();
        $request->prefer('return=representation');
        $request->body = [
            "intent" => "CAPTURE",
            "purchase_units" => [[
                "amount" => [
                    "currency_code" => "USD",
                    "value" => round($amount,2),
                ],
                "description" => getPaymentDetails($customerInfo['type'], $customerInfo),
            ]],
            "application_context" => [
                "cancel_url" => route('payment.error'),
                "return_url" => route('payment.success.paypal'),
            ],
        ];

        $response = $this->client->execute($request);

        $approvalUrl = collect($response->result->links)
            ->firstWhere('rel', 'approve')
            ->href;

        // Store order ID & customer info in session
        Session::put('paypal_order_id', $response->result->id);
        Session::put('customer_info', $customerInfo);

        return redirect($approvalUrl);
    }

    // Capture payment after approval
    public function capturePayment($request)
    {
        $orderId = $request->query('token') ?? Session::get('paypal_order_id');
        $customerInfo = Session::get('customer_info');

        if (!$orderId) {
            return redirect()->route('payment.error')->with('error',translate('Payment token missing.'));
        }

        $captureRequest = new OrdersCaptureRequest($orderId);
        $captureRequest->prefer('return=representation');

        $response = $this->client->execute($captureRequest);

        if ($response->statusCode == 201 && $response->result->status == 'COMPLETED') {
            Session::forget('paypal_order_id');
            Session::forget('customer_info');
            return ['status' => true, 'data' => $response->result, 'customer_info' => $customerInfo];
        }

        return back()->with('error', translate('Payment not completed.'));
    }
}
