<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Location;
use App\Models\Transaction;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Artist;
use App\Models\Order;
use App\Models\Wishlist;
use App\Models\Artwork;
use Illuminate\Support\Str;

class UserController extends Controller
{
    public function index(){
        $title = translate('User Dashboard');
        $user = Auth::user();
        $data['total_auction'] = Order::where('type',1)->where('user_id',$user->id)->count();
        $data['total_order'] = Order::where('type',2)->where('user_id',$user->id)->count();
        $data['total_deposit'] = Transaction::where('type',1)->where('user_id',$user->id)->count();
        $data['total_wishlist'] = Wishlist::where('user_id',$user->id)->count();
        return view('frontend.template-'.selectedTheme().'.pages.dashboard.index',compact('title','data'));
    }

    public function profile(){
        $title = translate('User Profile');
        $user = Auth::user();
        $countries = Location::whereNull('country_id')->whereNull('state_id')->orderBy('name','asc')->get();
        return view('frontend.template-'.selectedTheme().'.pages.dashboard.profile',compact('title','user','countries'));
    }

    public function profile_update(Request $request){
        $user = Auth::user();
        $validator = Validator::make($request->all(), [
            'fname' => 'required|string|max:255',
            'lname' => 'required|string|max:255',
            'address' => 'nullable|string|max:255',
            'email' => 'nullable|unique:users,email,'.auth()->id(),
            'country_id' => 'nullable|integer|exists:locations,id',
            'state_id' => 'nullable|integer|exists:locations,id',
            'city_id' => 'nullable|integer|exists:locations,id',
            'zip_code' => 'nullable|string|max:20',
            'phone' => 'nullable|max:20|unique:users,phone,' . $user->id,
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        if($request->hasFile('image')){
            $validator = Validator::make($request->all(), [
                'image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
            ]);
         }
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            // Delete old image if exists
            if ($user->image && file_exists(file_path() . $user->image)) {
                @unlink(file_path() . $user->image);
            }

            $user->image = uploadImage($image);
        }

        $user->fname = $request->fname;
        $user->lname = $request->lname;
        $user->address = $request->address;
        $user->email = $request->email;
        $user->country_id = $request->country_id;
        $user->state_id = $request->state_id;
        $user->city_id = $request->city_id;
        $user->zip_code = $request->zip_code;
        $user->phone = $request->phone;
        if($request->password){
            $user->password = Hash::make($request->password);
        }
        $user->update();

        return redirect()->back()->with('success',translate('Profile Updated Successfully'));
    }

    public function bidding(Request $request){
        $title = translate('All Bidding Items');
        $user = Auth::user();
        $biddings = Order::with('artworks') // always eager load to avoid N+1
            ->where('type', 1)
            ->where('user_id', $user->id)
            ->when($request->filter_by == 1, fn($q) => $q->orderBy('status'))
            ->when($request->filter_by == 2, fn($q) =>
                $q->leftJoin('artworks', 'orders.artwork_id', '=', 'artworks.id')
                ->orderBy('artworks.name')
                ->select('orders.*') // to avoid column ambiguity
            )
            ->latest('orders.created_at')
            ->paginate(10);
        return view('frontend.template-'.selectedTheme().'.pages.dashboard.bidding',compact('title', 'biddings'));
    }

    public function orders(Request $request){
        $title = translate('All Orders');
        $user = Auth::user();
        $orders = Order::with('artworks') // always eager load to avoid N+1
            ->where('type', 2)
            ->where('user_id', $user->id)
            ->when($request->filter_by == 1, fn($q) => $q->orderBy('status'))
            ->when($request->filter_by == 2, fn($q) =>
                $q->leftJoin('artworks', 'orders.artwork_id', '=', 'artworks.id')
                ->orderBy('artworks.name')
                ->select('orders.*') // to avoid column ambiguity
            )
            ->latest('orders.created_at')
            ->paginate(10);
        return view('frontend.template-'.selectedTheme().'.pages.dashboard.orders',compact('title','orders'));
    }


    public function wishlist(Request $request){
        $title = translate('All Wishlists');
        $user = Auth::user();
        $wishlists = Wishlist::when($request->filter_by, function ($q) use ($request) {
        return $q->leftJoin('artworks', 'artworks.id', '=', 'wishlists.artwork_id')
                 ->where('artworks.sale_type', $request->filter_by);
    })->select('wishlists.*')->where('wishlists.user_id',$user->id)->latest('wishlists.created_at')->paginate(10);
        return view('frontend.template-'.selectedTheme().'.pages.dashboard.wishlist',compact('title','wishlists'));
    }

    public function deposit(){
        $title = translate('All Deposits');
        $user = Auth::user();
        $deposits = Transaction::where('type',1)->where('user_id',$user->id)->latest()->paginate(10)->withQueryString();
        return view('frontend.template-'.selectedTheme().'.pages.dashboard.deposit',compact('title','deposits'));
    }

    public function panel_status(Request $request) {
        try {
            $user = Auth::user();
            $exists = Artist::where('user_id', $user->id)->latest()->first();
            if($exists){
                if($exists->status == 1 && empty($request->is_dashboard)){
                    $exists->status = 2;
                }else{
                    $exists->status = 1;
                }
                $exists->update();

                if($request->is_dashboard){
                    $artworks = Artwork::where('author_id',$user->id)->where('status',1)->get();
                    foreach($artworks as $artwork) {
                        $artwork->status = 2;
                        $artwork->update();
                    }
                }
            }else{
                $artist = new Artist;
                $artist->user_id = $user->id;
                $artist->name = $user->fname.' '.$user->lname;
                $slug = Str::slug($user->fname, '-');
                $same_slug_count = Artist::where('slug', 'LIKE', $slug . '%')->count();
                $slug_suffix = $same_slug_count ? '-' . $same_slug_count + 1 : '';
                $slug .= $slug_suffix;
                $artist->slug = $slug;
                $artist->status = 1;
                $artist->save();
            }

            if($user->isDashboard == 1 && empty($request->is_dashboard)){
                $user->isDashboard = 2;
            }else{
                $user->isDashboard = 1;
            }
            $user->update();

            return redirect()->back()
            ->with('success', translate('Dashboard Panel status changed successfully'));
        } catch (\Throwable $th) {
             return redirect()->back()
            ->with('error', $th->getMessage());
        }
    }
    /**
     * order_details
     *
     * @param  mixed $request
     * @param  int $id
     * @return View
     */
    public function order_details(Request $request, $id)
    {

        $lang = $request->lang;
        $user = Auth::user();
        try {
            $bidSingle = Order::with('transactions')->where('id', $id)->first();

            $title = $bidSingle->type == 1 ? translate('Bid Details') : translate('Order Details');
            $meta_description = get_setting('meta_description');
            $meta_keyward = get_setting('meta_keyward');
            $meta_image = get_setting('header_logo') ? uploaded_asset(get_setting('header_logo')) : '';
            $review_confirm = 10;

            return view('frontend.template-' . selectedTheme() . '.pages.dashboard.order_details', compact('title', 'meta_image', 'meta_description', 'meta_keyward', 'bidSingle', 'lang', 'review_confirm'));
        } catch (\Throwable $th) {
            return view('frontend.error.index');
        }
    }

    /**
     * wishlist_store
     *
     * @param  mixed $request
     * @return View
     */
    public function wishlist_store(Request $request) {
        $user = auth()->user();
        $itemId = $request->item_id;

        // Check if already added
        $exists = $user->wishlists()->where('artwork_id', $itemId)->exists();

    if ($exists) {
        return response()->json([
            'status' => 'exists',
            'message' => translate('Artwork already added in the Wishlist')
        ]);
    }

    // Save new record
    $user->wishlists()->create([
        'artwork_id' => $itemId,
    ]);

    return response()->json([
        'status' => 'added',
        'message' => translate('The Artwork added to wishlist successfully')
    ]);

    }

    /**
     * wishlist_destroy
     *
     * @param  mixed $id
     * @return View
     */
    public function wishlist_destroy($id){
        Wishlist::findOrFail($id)->delete();
        return redirect()->back()
            ->with('success', translate('Wishlist deleted successfully'));
    }
}
