<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SupportTicket;
use Illuminate\Support\Facades\Auth;
use App\Models\SupportTicketAttachment;
use App\Models\TicketReply;
use App\Models\TicketReplyDocument;
use Illuminate\Support\Facades\Validator;
use Mews\Purifier\Facades\Purifier;

class SupportTicketController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
{
    $user = Auth::user();
    $page_title = translate('Ticket & Support');

    $query = SupportTicket::where('parent_id', 0);
    if ($user->role != 3) {
        $query->where('user_id', $user->id);
    }

    $supports = $query->latest()->paginate(10);
    $baseQuery = clone $query;

    $answered = TicketReply::whereIn('support_ticket_id', $query->pluck('id'))->count();
    $opened   = (clone $baseQuery)->where('status', 1)->count();
    $closed   = (clone $baseQuery)->where('status', 2)->count();
    $total    = $baseQuery->count();

    return view('backend.support_ticket.index', compact('page_title', 'supports', 'answered', 'opened', 'closed', 'total'));
}


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (Auth::user()->role != 3) {
            $page_title = translate('Add Ticket');
            return view('backend.support_ticket.create', compact('page_title'));
        } else {
            return redirect()->route('dashboard.supports.index')->with('error', translate('Support Ticket will be open only User'));
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $validator = Validator::make($request->all(), [
            'subject' => 'required|max:255',
            'description' => 'nullable',
            'department' => 'required|max:255',
            'priority' => 'required|max:255',
            'attachment' => 'nullable',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $supports = new SupportTicket;
        $supports->subject = $request->subject;
        $supports->department = $request->department;
        $supports->priority = $request->priority;
        $supports->description = Purifier::clean($request->description);
        $supports->type = 1;
        $supports->user_id = $user->id;
        if ($supports->save()) {
            $support_id = $supports->id;
            if ($request->attachment != null) {
                foreach ($request->attachment as $key => $val) {
                    $attachments = new SupportTicketAttachment;
                    $attachments->attachment = uploadImage($val);
                    $attachments->support_ticket_id = $support_id;
                    $attachments->save();
                }
            }
        }

        return redirect()->route('dashboard.supports.index')->with('success', translate('Support Ticket opened successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $page_title = translate('Reply Ticket');
        $supports = SupportTicket::with('users', 'support_attachments', 'ticketReplies.documents', 'ticketReplies.authorInfo')->findOrFail($id);
        return view('backend.support_ticket.reply', compact('page_title', 'supports'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'answer' => 'required',
        ], [
            'answer.required' => 'Reply Answer is Required'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }
        $ticketReply =  new TicketReply();
        $ticketReply->support_ticket_id = $id;
        $ticketReply->author_reply_id = Auth::user()->id;
        $ticketReply->answer = prelaceScript(html_entity_decode($request->answer)) ;
        $ticketReply->save();
        if ($request->attachment != null) {
            foreach ($request->attachment as $key=> $val) {
                $attachment = new TicketReplyDocument;
                $attachment->document = uploadImage($val);
                $attachment->ticket_reply_id = $ticketReply->id;
                $attachment->save();
            }
        }
        return redirect()->back()->with('success', translate('Support Ticket reply successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $support = SupportTicket::with('support_attachments', 'ticketReplies.documents')->findOrFail($id);
        if (!empty($support->support_attachments)) {
            foreach ($support->support_attachments as $sup_attach) {
                if ($sup_attach->attachment && file_exists(file_path().$sup_attach->attachment)) {
                    unlink(file_path() . $sup_attach->attachment);
                }
            }
        }

        if (!empty($support->ticketReplies)) {
            foreach ($support->ticketReplies as $ticketReply) {
                if (!empty($ticketReply->documents)) {
                    foreach ($ticketReply->documents as $file) {
                        if ($file->document && file_exists(file_path().$file->document)) {
                            unlink(file_path() . $file->document);
                        }
                    }
                }
            }
        }


        $support->delete();
        return back()->with('success', translate('Support Ticket deleted successfully'));
    }



    /**
     * closeTicket
     *
     * @param  int $supportId
     * @return Response
     */
    public function closeTicket($supportId)
    {
        $support = SupportTicket::where('id', $supportId)->first();
        $support->status = 2;
        if ($support->update()) {
            return redirect()->route('dashboard.supports.index')->with('success', translate('Support Ticket Close successfully'));
        }
    }
}
