<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\Artwork;
use Illuminate\Support\Facades\Auth;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
public function index(Request $request)
{
    $user = Auth::user();
    $page_title = translate('All Orders');
    $lang = $request->lang;
    $search = $request->search;

    $query = Order::select('orders.*')
        ->join('artworks', 'artworks.id', '=', 'orders.artwork_id');

    if ($user->role != 3) {
        $query->where('artworks.author_id', $user->id);
    }

    if ($search) {
        $query->where('artworks.name', 'LIKE', "%{$search}%");
    }

    $orders = $query->where('orders.type', 2)->latest('orders.created_at')->paginate(10);

    $authorId = $user->role == 3 ? null : $user->id;
    $data = [
        'total_sale_amount' => $this->salesSubByStatus(2, 3, $authorId),
        'total_orders' => $this->allOrder($authorId)->count(),
        'total_processing_orders' => $this->getOrderByStatus(1, $authorId)->count(),
        'total_completed_orders' => $this->getOrderByStatus(4, $authorId)->count(),
        'total_deliverd_orders' => $this->getOrderByStatus(6, $authorId)->count(),
    ];

    return view('backend.orders.index', compact('page_title', 'orders', 'lang', 'data'));
}

/**
     * Display a listing of the Bidding.
     */
public function bidding(Request $request)
{
    $user = Auth::user();
    $page_title = translate('All Bidding');
    $lang = $request->lang;

    $query = Order::select('orders.*')
        ->join('artworks', 'artworks.id', '=', 'orders.artwork_id')
        ->when($request->search, fn($q) =>
            $q->where('artworks.name', 'LIKE', "%{$request->search}%")
                ->whereNotIn('orders.status', [2, 4, 6, 8])
        );

    if ($user->role == 3) {
        $data = [
            'total_bidder'         => Order::where('type', 1)->count(),
            'total_bidding_amount' => Order::where('type', 1)->where('payment_status', 3)->sum('amount'),
            'total_winner'         => Order::where('type', 1)->where('win_status', 1)->count(),
            'live_auction'         => Artwork::where(['sale_type' => 1, 'status' => 1])->count(),
            'closed_bids'          => Artwork::where(['sale_type' => 1, 'status' => 5])->count(),
        ];
    } else {
        $query->where('artworks.author_id', $user->id);

        $data = [
            'total_bidder'         => Order::join('artworks', 'artworks.id', '=', 'orders.artwork_id')
                ->where(['orders.type' => 1, 'artworks.author_id' => $user->id])
                ->count(),

            'total_bidding_amount' => Order::join('artworks', 'artworks.id', '=', 'orders.artwork_id')
                ->where(['orders.type' => 1, 'orders.payment_status' => 3, 'artworks.author_id' => $user->id])
                ->sum('orders.amount'),

            'total_winner'         => Order::join('artworks', 'artworks.id', '=', 'orders.artwork_id')
                ->where(['orders.type' => 1, 'orders.win_status' => 1, 'artworks.author_id' => $user->id])
                ->count(),

            'live_auction'         => Artwork::where(['sale_type' => 1, 'status' => 1, 'author_id' => $user->id])->count(),
            'closed_bids'          => Artwork::where(['sale_type' => 1, 'status' => 5, 'author_id' => $user->id])->count(),
        ];
    }

    $bidding = $query->where('orders.type', 1)->latest('orders.created_at')->paginate(10);

    return view('backend.orders.bidding', compact('page_title', 'bidding', 'lang', 'data'));
}


    /**
     * Change order Status.
     */
    public function changeStatus(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->status = $request->status_id;
        $order->update();
        return redirect()->back()->with('success', translate('Order status changed successfully!'));
    }

    /**
     * Display the specified resource.
     */
    public function details(Request $request, $id)
    {
        $page_title = translate('Invoice');
        $lang = $request->lang;
        $order = Order::findOrFail($id);
        $order->increment('view', 1);
        return view('backend.orders.details', compact('page_title', 'order', 'lang'));
    }


/**
 * Get orders by status and optionally filter by author (artwork creator)
 */
public function getOrderByStatus($status, $authorId = null)
{
    $orders = Order::where('type', 2)->where('status', $status);

    if (!empty($authorId)) {
        $orders->whereHas('artworks', function ($query) use ($authorId) {
            $query->where('author_id', $authorId);
        });
    }

    return $orders->get();
}

/**
 * Get all orders, optionally filtering by artwork author
 */
public function allOrder($authorId = null)
{
    $orders = Order::where('type', 2);

    if (!empty($authorId)) {
        $orders->whereHas('artworks', function ($query) use ($authorId) {
            $query->where('author_id', $authorId);
        });
    }

    return $orders->get();
}

/**
 * Calculate total sales amount by order type, payment status, and optionally artwork author
 */
public function salesSubByStatus($type = null, $status = null, $authorId = null)
{
    $orders = Order::where('type', $type)->where('payment_status', $status);

    if (!empty($authorId)) {
        $orders->whereHas('artworks', function ($query) use ($authorId) {
            $query->where('author_id', $authorId);
        });
    }

    return $orders->sum('amount');
}
}
