<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Artist;
use App\Models\Location;
use App\Models\Order;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CustomerController extends Controller
{
    public function index(Request $request)
    {
        $page_title = translate('Customer List');

        $customers = User::where('role', 1)
            ->when($request->search, function ($query) use ($request) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where(DB::raw("concat(fname, ' ', lname)"), 'LIKE', "%{$search}%")
                      ->orWhere('email', 'LIKE', "%{$search}%")
                      ->orWhere('custom_id', $search);
                });
            })
            ->latest()
            ->paginate(12);

        $data = [
            'total_customers' => User::where('role', 1)->count(),
            'total_deposit'   => Transaction::where('type',1)->sum('total_amount'),
            'total_pay'       => Transaction::where('type',1)->where('status',2)->sum('total_amount'),
            'total_pending'   => Transaction::where('type',1)->where('status',1)->sum('total_amount'),
        ];

        return view('backend.customer.index', compact('page_title', 'customers', 'data'));
    }

    public function create()
    {
        return view('backend.customer.create', [
            'page_title' => translate('Create Customer'),
            'countries'  => Location::whereNull('country_id')->whereNull('state_id')->get(),
        ]);
    }

    public function store(Request $request)
    {
        $this->validateCustomer($request);

        $customer = new User($request->except('password', 'image'));
        $customer->password = Hash::make($request->password);
        $customer->role = 1;

        $customer->custom_id = generate_custom_id(1);

        if ($image = $request->file('image')) {
            $customer->image = uploadImage($image);
        }
        
        if($customer->save()){
            $artists = new Artist;
            $artists->name = $customer->fname.' '.$customer->lname;
            $artists->user_id = $customer->id;
            $artists->status = 2;
            $artists->save();
        }

        return redirect()->route('dashboard.customer.index')->with('success', translate('Customer saved successfully'));
    }

    public function show($id)
    {
        $page_title = translate('Customer Profile');
        $customerSingle = User::findOrFail($id);
        $countries = Location::whereNull('country_id')->whereNull('state_id')->get();
        $deposits = $customerSingle->deposits;
        $orders = Order::where('user_id', $customerSingle->id)->latest()->limit(15)->get();
        $data['deposits'] = $customerSingle->deposits?->sum('total_amount');
        $data['bids'] = $customerSingle->bids?->count();
        $data['orders'] = $customerSingle->orders?->count();
        return view('backend.customer.profile', compact('page_title', 'customerSingle', 'countries', 'data', 'deposits', 'orders'));
    }

    public function edit($id)
    {
        return view('backend.customer.edit', [
            'page_title'     => translate('Edit Customer'),
            'customerSingle' => User::findOrFail($id),
            'countries'      => Location::whereNull('country_id')->whereNull('state_id')->get(),
        ]);
    }

    public function update(Request $request, $id)
    {
        $customer = User::findOrFail($id);

        $this->validateCustomer($request, $id);

        $customer->fill($request->except('password', 'image'));

        if ($request->filled('password')) {
            $customer->password = Hash::make($request->password);
        }

        if ($image = $request->file('image')) {
            if ($customer->image && file_exists(file_path().$customer->image)) {
                unlink(file_path().$customer->image);
            }
            $customer->image = uploadImage($image);
        }

        $customer->save();

        return redirect()->route('dashboard.customer.index')->with('success', translate('Customer has been updated successfully'));
    }

    public function destroy($id)
    {
        $customer = User::findOrFail($id);
        if ($customer->image && file_exists(file_path().$customer->image)) {
            unlink(file_path().$customer->image);
        }
        $customer->delete();

        return back()->with('success', translate('Customer deleted successfully'));
    }

    public function changeStatus(Request $request)
    {
        $data_id = $request->data_id;
        if ($data_id) {
            $customer = User::findOrFail($data_id);
            if ($customer->status == 1) {
                $customer->status = 2;
                $message = translate('Deactive');
            } else {
                $customer->status = 1;
                $message = translate('Active');
            }
            if ($customer->update()) {
                $response = array('output' => 'success', 'statusId' => $customer->status, 'dataId' => $customer->id, 'message' => $message);
                return response()->json($response);
            }
        }
    }

    public function login($id)
    {
        auth()->login(User::findOrFail(decrypt($id)), true);
        return redirect()->route('user.dashboard');
    }

    // ---------- Private Utility Methods ----------

    private function validateCustomer(Request $request, $id = null)
    {
        $uniqueEmail = 'unique:users,email' . ($id ? ",$id" : '');
        $uniquePhone = 'unique:users,phone' . ($id ? ",$id" : '');

        $rules = [
            'fname'      => 'required|max:255',
            'lname'      => 'required|max:255',
            'email'      => "required|max:255|$uniqueEmail",
            'phone'      => "required|max:255|$uniquePhone",
            'address'    => 'required|max:255',
            'country_id' => 'required',
            'state_id'   => 'required',
            'city_id'    => 'required',
            'zip_code'   => 'required|max:255',
            'password'   => ($id ? 'nullable' : 'required') . '|confirmed|min:8',
            'image'      => 'nullable|image',
        ];

        Validator::make($request->all(), $rules)->validate();
    }

}

