<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Artist;
use App\Models\Artwork;
use App\Models\Order;
use App\Models\Category;
use App\Models\Transaction;
use Illuminate\Support\Str;
use App\Models\ArtworkGallery;
use App\Models\ArtworkTranslation;
use Mews\Purifier\Facades\Purifier;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ArtworkController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $lang = $request->lang;
        $page_title = translate('All Artworks');
        if ($user->role == 3) {
                $artworks = Artwork::when($request->has('search'), function ($query) use ($request) {
                        $query->where('name', 'LIKE', '%' . $request->search . '%')
                        ->orWhere('sale_price', $request->search)
                        ->orWhere('price', $request->search)
                        ->orWhere('min_bid_price', $request->search);
                    })->latest()->paginate(10);

            $data['total_products'] = $this->get_artworks();
            $data['total_auction_products'] = $this->get_artworks(1);
            $data['total_direct_products'] = $this->get_artworks(2);
        } else {
                $artworks = Artwork::where('author_id',$user->id)
                ->when($request->has('search'), function ($query) use ($request) {
                        $query->where('name', 'LIKE', '%' . $request->search . '%')
                        ->orWhere('sale_price', $request->search);
                    })->latest()->paginate(10);

            $data['total_products'] = $this->get_artworks('', $user->id);
            $data['total_auction_products'] = $this->get_artworks(1, $user->id);
            $data['total_direct_products'] = $this->get_artworks(2, $user->id);
        }
        return view('backend.artwork.index', compact('page_title', 'artworks', 'lang', 'data'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();
        $page_title = translate('Create Artwork');
        $categories = Category::where('status', 1)->orderBy('name', 'asc')->get();
        if($user->role == 3){
            $authors = User::where('role', 2)->orderBy('fname', 'asc')->get();
        }else{
            $authors = Artist::where('user_id', $user->id)->orderBy('name', 'asc')->get();
        }

        return view('backend.artwork.create', compact('categories', 'authors', 'page_title'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $user =  Auth::user();
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
            'content' => 'required',
            'tags' => 'nullable',
            'height' => 'nullable|max:255',
            'width' => 'nullable|max:255',
            'length' => 'nullable|max:255',
            'year' => 'nullable|max:255',
            'location' => 'nullable|max:255',
            'condition' => 'required|max:255',
            'specifications' => 'nullable|array',
            'enable_seo' => 'nullable|max:255',
            'meta_title' => 'nullable|max:255',
            'meta_keyward' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_image' => 'nullable|max:255',
            'author_id' => 'nullable|max:255',
            'artist_id' => 'nullable|max:255',
            'category_id' => 'required|array',
            'quantity' => 'required|max:255',
            'sale_type' => 'required|max:255',
            'min_deposit' => 'nullable|max:255',
            'min_deposit_type' => 'nullable|max:255',
            'price' => 'nullable|max:255',
            'sale_price' => 'nullable|max:255',
            'min_bid_price' => 'nullable|max:255',
            'schedule_type' => 'nullable|max:255',
            'start_date' => 'nullable|max:255',
            'end_date' => 'nullable|max:255',
            'status' => 'nullable|max:255',
            'features_image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
            'image' => 'nullable',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $artwork = new Artwork();

        if ($request->hasFile('features_image')) {
            $artwork->features_image = uploadImage($request->file('features_image'));
        }

        if ($request->hasFile('meta_image')) {
            $artwork->meta_image = uploadImage($request->file('meta_image'));
        }
        $artwork->name = $request->name;

        $slug = Str::slug($request->name, '-');
        $same_slug_count = Artwork::where('slug', 'LIKE', $slug . '%')->count();
        $slug_suffix = $same_slug_count ? '-' . $same_slug_count + 1 : '';
        $slug .= $slug_suffix;
        $artwork->slug = $slug;

        $artwork->certificate_number = $request->certificate_number;
        $artwork->content = Purifier::clean($request->content);
        $artwork->tags = $request->tags;
        $artwork->height = $request->height;
        $artwork->width = $request->width;
        $artwork->length = $request->length;
        $artwork->year = $request->year;
        $artwork->location = $request->location;
        $artwork->condition = $request->condition;
        $artwork->enable_seo = $request->enable_seo == "on" ? 1 : 2;
        $artwork->meta_title = $request->meta_title;
        $artwork->meta_keyward = $request->meta_keyward;
        $artwork->meta_description =  Purifier::clean($request->meta_description);
        $artwork->quantity = $request->quantity;
        $artwork->sale_type = $request->sale_type;
        $artwork->min_deposit = $request->min_deposit;
        $artwork->min_deposit_type = $request->min_deposit_type;
        $artwork->price = $request->price;
        $artwork->sale_price = $request->sale_price;
        $artwork->min_bid_price = $request->min_bid_price;
        $artwork->schedule_type = $request->schedule_type;
        if($request->schedule_type == 1){
            $artwork->start_date = $request->start_date ? date('Y-m-d', strtotime($request->start_date)) : '';
        }else{
            $artwork->start_date = date('Y-m-d', strtotime(now()));
        }
        $artwork->end_date = $request->end_date ? date('Y-m-d', strtotime($request->end_date)) : '';
        $artwork->specifications = json_encode($request->specifications);
        if($user->role == 1){
            $artwork->artist_id = $user->artists?->first()?->id;
            $artwork->author_id = $user->id;
            $artwork->status = 3;
        }else if($user->role == 3){
            $artwork->author_id = $request->author_id;
            $artwork->artist_id = $request->artist_id;
            $artwork->status = $request->status;
        }else{
            $artwork->artist_id = $request->artist_id;
            $artwork->author_id = $user->id;
            $artwork->status = 3;
        }
        if($artwork->save()){
            $artwork->categories()->sync($request->category_id);
            $this->productGallery($request, $artwork->id);
        }

        return redirect()->route('dashboard.artworks.index')->with('success', translate('Artwork saved successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request,$id)
    {
        $page_title = translate('Edit Artwork');
        $lang = $request->lang;
        $artworkSingle = Artwork::findOrFail($id);
        $categories = Category::where('status', 1)->orderBy('name', 'asc')->get();
        return view('backend.artwork.edit', compact('categories', 'artworkSingle', 'page_title','lang'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $user =  Auth::user();
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
            'content' => 'required',
            'tags' => 'nullable',
            'height' => 'nullable|max:255',
            'width' => 'nullable|max:255',
            'length' => 'nullable|max:255',
            'year' => 'nullable|max:255',
            'location' => 'nullable|max:255',
            'condition' => 'required|max:255',
            'specifications' => 'nullable|array',
            'enable_seo' => 'nullable|max:255',
            'meta_title' => 'nullable|max:255',
            'meta_keyward' => 'nullable|max:255',
            'meta_description' => 'nullable',
            'meta_image' => 'nullable|max:255',
            'category_id' => 'required|array',
            'quantity' => 'required|max:255',
            'sale_type' => 'required|max:255',
            'min_deposit' => 'nullable|max:255',
            'min_deposit_type' => 'nullable|max:255',
            'price' => 'nullable|max:255',
            'sale_price' => 'nullable|max:255',
            'min_bid_price' => 'nullable|max:255',
            'schedule_type' => 'nullable|max:255',
            'start_date' => 'nullable|max:255',
            'end_date' => 'nullable|max:255',
            'status' => 'nullable|max:255',
            'image' => 'nullable|array',
        ]);

        if($request->hasFile('features_image')){
            $validator = Validator::make($request->all(), [
                'features_image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
            ]);
         }

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $artwork = Artwork::findOrFail($id);

        if ($request->hasFile('features_image')) {
            if ($artwork->features_image && file_exists(file_path().$artwork->features_image)) {
                unlink(file_path().$artwork->features_image);
            }
            $artwork->features_image = uploadImage($request->file('features_image'));
        }

        if ($request->hasFile('meta_image')) {
            if ($artwork->meta_image && file_exists(file_path().$artwork->meta_image)) {
                unlink(file_path().$artwork->meta_image);
            }
            $artwork->meta_image = uploadImage($request->file('meta_image'));
        }

        if ($request->lang == get_setting("DEFAULT_LANGUAGE", "en")) {
            $artwork->name = $request->name;
            $artwork->content = Purifier::clean($request->content);
            $artwork->specifications = json_encode($request->specifications);
            $artwork->tags = $request->tags;
        }

        $artwork->certificate_number = $request->certificate_number;
        $artwork->height = $request->height;
        $artwork->width = $request->width;
        $artwork->length = $request->length;
        $artwork->year = $request->year;
        $artwork->location = $request->location;
        $artwork->condition = $request->condition;
        $artwork->enable_seo = $request->enable_seo == "on" ? 1 : 2;
        $artwork->meta_title = $request->meta_title;
        $artwork->meta_keyward = $request->meta_keyward;
        $artwork->meta_description = $request->meta_description;
        $artwork->quantity = $request->quantity;
        $artwork->sale_type = $request->sale_type;
        $artwork->min_deposit = $request->min_deposit;
        $artwork->min_deposit_type = $request->min_deposit_type;
        $artwork->price = $request->price;
        $artwork->sale_price = $request->sale_price;
        $artwork->min_bid_price = $request->min_bid_price;
        $artwork->schedule_type = $request->schedule_type;
        if($request->schedule_type == 1){
            $artwork->start_date = $request->start_date ? date('Y-m-d', strtotime($request->start_date)) : '';
        }else{
            $artwork->start_date = date('Y-m-d', strtotime(now()));
        }
        $artwork->end_date = $request->end_date ? date('Y-m-d', strtotime($request->end_date)) : '';
        $artwork->specifications = json_encode($request->specifications);
        if($user->role == 3){
            $artwork->status = $request->status;
        }else{
            $artwork->status = 3;
        }
        if($artwork->update()){
        $artwork->categories()->sync($request->category_id);
        $this->productGallery($request, $artwork->id);

            $artwork_translation = ArtworkTranslation::firstOrNew(['lang' => $request->lang, 'artwork_id' => $artwork->id]);
            $artwork_translation->name = $request->name;
            $artwork_translation->content = Purifier::clean($request->content);
            $artwork_translation->specifications = json_encode($request->specifications);
            $artwork_translation->tags = $request->tags;
            $artwork_translation->save();

        }

        return redirect()->back()->with('success', translate('Artwork has been updated successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $artworks = Artwork::findOrFail($id);

        if ($artworks->features_image && file_exists(file_path().$artworks->features_image)) {
            unlink(file_path() . $artworks->features_image);
        }

        if ($artworks->meta_image && file_exists(file_path().$artworks->meta_image)) {
            unlink(file_path() . $artworks->meta_image);
        }

        $artworks->delete();
        return back()->with('success', translate('Artwork deleted successfully'));
    }

     /**
     * Change Category status.
     */

    public function changeStatus(Request $request)
    {
        $data_id     = $request->data_id;
        if ($data_id) {
            $artworks = Artwork::findOrFail($data_id);
            if ($artworks->status == 1) {
                $artworks->status = 2;
                $message = translate('Artwork Inactive');
            } else {
                $artworks->status = 1;
                $message = translate('Artwork Active');
            }
            if ($artworks->update()) {
                $response = array('output' => 'success', 'statusId' => $artworks->status, 'dataId' => $artworks->id, 'message' => $message);
                return response()->json($response);
            }
        }
    }

    public function get_artists(Request $request)
    {
        $data['artists'] = Artist::where('user_id', $request->author_id)->get(['name', 'id']);
        $data['option'] = translate('Select Option');
        return response()->json($data);
    }

    /**
     * productGallery
     *
     * @param  mixed $request
     * @param  int $artwork_id
     * @return Response
     */
    public function productGallery($request, $artwork_id)
    {
        if ($request->file('image')) {
            $allowedfileExtension=['jpeg','png','jpg','gif','svg','webp'];
            foreach ($request->file('image') as $image) {
                $extension = $image->getClientOriginalExtension();
                $check = in_array($extension,$allowedfileExtension);
                if($check){
                    $artwork_gallery = new ArtworkGallery;
                    $artwork_gallery->image = uploadImage($image);
                    $artwork_gallery->artwork_id = $artwork_id;
                    $artwork_gallery->save();
                }
            }
        }
    }

    /**
     * Gallery image remove.
     */

    public function gallery_remove()
    {
        $dataId = $_POST['dataId'];

        if ($dataId) {
            $gallery = ArtworkGallery::findOrFail($dataId);
            if (file_exists(file_path().$gallery->image)) {
                unlink(file_path() . $gallery->image);
            }

            $message = translate('Gallery Image Deleted');

            if ($gallery->delete()) {
                $response = array('output' => 'success', 'dataId' => $dataId, 'message' => $message);
                return response()->json($response);
            }
        }
    }

    public function get_artworks($type = NULL, $user_id = NULL){
        $artworks = Artwork::when($type, function ($query) use ($type) {
                        $query->where('sale_type', $type);
                    })->when($user_id, function ($query) use ($user_id) {
                        $query->where('author_id', $user_id);
                    })->count();
        return $artworks;
    }

    /**
     * Display the specified resource.
     */
    public function details(Request $request, $id)
    {
        $page_title = translate('Artwork Details');
        $lang = $request->lang;
        $artworkSingle = Artwork::with('bids')->where('id', $id)->first();
        return view('backend.artwork.details', compact('page_title', 'artworkSingle', 'lang'));
    }

     /**
     * closed
     *
     * @param  mixed $id
     * @return void
     */
    public function closed($id)
{
    $artwork = Artwork::findOrFail($id);
    $winner = Order::where('artwork_id', $artwork->id)->where('win_status', 1)->first();

    if (!$winner) {
        return back()->with('error', translate('Please confirm your winner'));
    }

    $artwork->status = 5;
    $orders = Order::where('artwork_id', $artwork->id)->get();

    foreach ($orders as $order) {
        $transaction = Transaction::where('order_id', $order->id)->first();

        if ($order->status == 1 && $transaction) {
            $transaction->update([
                'type' => 6,
                'payment_details' => 'Bidding Refund',
                'status' => 3
            ]);
            $order->update(['status' => 7]);
            User::find($order->user_id)?->increment('wallet_balance', $order->amount);
        }

        if ($order->win_status == 1 && $artwork->author?->role == 2 && $transaction) {
            $transaction->update([
                'payment_details' => 'Bid winning',
                'status' => 2
            ]);
            User::find($artwork->author_id)?->increment('wallet_balance', (int) $transaction->merchant_amount);
        }
    }

    $artwork->save();
    return back()->with('success', translate('Bid Closed successfully'));
}


    /**
     * Pending product approved.
     */
    public function approve(Request $request, $id)
    {
        $artworks = Artwork::findOrFail($id);
        $artworks->status = 1;
        $artworks->update();

        return back()->with('success', translate('Artwork approved successfully'));
    }


     /**
     * winner
     *
     * @param  mixed $id
     * @return void
     */
    public function winner($id)
    {
        $order = Order::with('users')->where('id', $id)->first();
        if ($order->win_status == 1) {
            $message = translate("Already Winner");
        } else {
            $order->win_status = 1;
            $order->status = 2;
            email_send('bid_winner', $order->users->email);
            $order->update();
            $message =  translate('Winner updated successfully');
        }

        return back()->with('success', $message);
    }

}
