<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Artist;
use App\Models\ArtistTranslation;
use App\Models\User;
use App\Models\Artwork;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Mews\Purifier\Facades\Purifier;

class ArtistController extends Controller
{
    /**
     * Display a listing of the artists.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $lang = $request->lang;
        $page_title = translate('Artist List');

        $artists = Artist::query()
        ->when($request->search, fn($q) => $q->where('name', 'like', '%' . $request->search . '%'))
        ->when($user->role != 3, fn($q) => $q->where('user_id', $user->id))
        ->latest()
        ->paginate(12);

        $baseQuery = $user->role != 3 ? Artist::where('user_id', $user->id) : Artist::query();

        $data = [
            'total_artist'    => (clone $baseQuery)->count(),
            'active_artist'   => (clone $baseQuery)->where('status', 1)->count(),
            'inactive_artist' => (clone $baseQuery)->where('status', 2)->count(),
            'total_artwork'   => Artwork::when($user->role != 3, fn($q) => $q->where('author_id', $user->id))->count(),
        ];


        return view('backend.artists.index', compact('page_title', 'artists', 'data', 'lang'));
    }

    /**
     * Show the form for creating a new artist.
     */
    public function create()
    {
        $page_title = translate('Create Artist');
        $merchants = User::where('status', 1)
            ->where('role', 2)
            ->orderBy('fname', 'asc')
            ->get();

        return view('backend.artists.create', compact('merchants', 'page_title'));
    }

    /**
     * Store a newly created artist in storage.
     */
    public function store(Request $request)
    {

        $user =  Auth::user();
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
            'bio' => 'nullable',
            'dob' => 'nullable|string|max:255',
            'nationality' => 'nullable|max:255',
            'signature' => 'nullable',
            'themes' => 'nullable',
            'biography_desc' => 'nullable',
            'career' => 'nullable|array',
            'awards_desc' => 'nullable',
            'awards' => 'nullable|array',
            'image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $artist = new Artist($request->except('slug', 'dob', 'career', 'awards', 'image', 'signature', 'merchant'));

        $slug = Str::slug($request->name, '-');
        $same_slug_count = Artist::where('slug', 'LIKE', $slug . '%')->count();
        $slug_suffix = $same_slug_count ? '-' . $same_slug_count + 1 : '';
        $slug .= $slug_suffix;
        $artist->slug = $slug;

        $artist->dob = (string)$request->dob;
        $artist->career_lists = json_encode($request->career);
        $artist->awards_lists = json_encode($request->awards);
        if($user->role == 2){
            $artist->user_id = $user->id;
        }else{
            $artist->user_id = $request->merchant;
        }

        if ($request->hasFile('image')) {
            $artist->image = uploadImage($request->file('image'));
        }
         if ($request->hasFile('signature')) {
            $artist->signature = uploadImage($request->file('signature'));
        }

        $artist->save();

        return redirect()->route('dashboard.artist.index')
            ->with('success', translate('Artist saved successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request,$id){
        $lang = $request->lang;
        $page_title = translate('Edit Artist');
        $artistSingle = Artist::findOrFail($id);

        return view('backend.artists.edit', compact('artistSingle', 'page_title', 'lang'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|max:255',
            'bio' => 'nullable',
            'dob' => 'nullable|string|max:255',
            'nationality' => 'nullable|max:255',
            'signature' => 'nullable',
            'themes' => 'nullable',
            'biography_desc' => 'nullable',
            'career' => 'nullable|array',
            'awards_desc' => 'nullable',
            'awards' => 'nullable|array',
        ]);

        if($request->hasFile('image')){
            $validator = Validator::make($request->all(), [
                'image' => 'required|mimes:jpeg,png,jpg,gif,svg,webp',
            ]);
         }
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $artist = Artist::findOrFail($id);
        if ($request->lang == get_setting("DEFAULT_LANGUAGE", "en")) {
            $artist->name = $request->name;
            $artist->bio = $request->bio;
            $artist->dob = $request->dob;
            $artist->nationality = $request->nationality;
            $artist->themes = $request->themes;
            $artist->biography_desc = $request->biography_desc;
            $artist->career_lists = json_encode($request->career);
            $artist->awards_desc = $request->awards_desc;
            $artist->awards_lists = json_encode($request->awards);
        }

        if ($image = $request->file('image')) {
            if ($artist->image && file_exists(file_path().$artist->image)) {
                unlink(file_path().$artist->image);
            }
            $artist->image = uploadImage($image);
        }
        if ($signature = $request->file('signature')) {
            if ($artist->signature && file_exists(file_path().$artist->signature)) {
                unlink(file_path().$artist->signature);
            }
            $artist->signature = uploadImage($signature);
        }

        if($artist->update()){
            $artist_translation = ArtistTranslation::firstOrNew(['lang' => $request->lang, 'artist_id' => $artist->id]);
            $artist_translation->name = $request->name;
            $artist_translation->bio = $request->bio;
            $artist_translation->dob = $request->dob;
            $artist_translation->nationality = $request->nationality;
            $artist_translation->themes = $request->themes;
            $artist_translation->biography_desc = $request->biography_desc;
            $artist_translation->career_lists = json_encode($request->career);
            $artist_translation->awards_desc = $request->awards_desc;
            $artist_translation->awards_lists = json_encode($request->awards);
            $artist_translation->save();
        }
        if(Auth::user()->role == 1){
            return redirect()->back()->with('success', translate('Artist has been updated successfully'));
        }else{
            return redirect()->route('dashboard.artist.index')->with('success', translate('Artist has been updated successfully'));
        }
    }

     /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $artists = Artist::findOrFail($id);
        if ($artists->image && file_exists(file_path().$artists->image)) {
            unlink(file_path() . $artists->image);
        }

        $artists->delete();
        return back()->with('success', translate('Artist deleted successfully'));
    }

    /**
     * Change Blog status.
     */

    public function changeStatus(Request $request)
    {
        $data_id     = $request->data_id;
        if ($data_id) {
            $artists = Artist::findOrFail($data_id);
            if ($artists->status == 1) {
                $artists->status = 2;
                $message = translate('Artist Inactive');
            } else {
                $artists->status = 1;
                $message = translate('Artist Active');
            }
            if ($artists->update()) {
                $response = array('output' => 'success', 'statusId' => $artists->status, 'dataId' => $artists->id, 'message' => $message);
                return response()->json($response);
            }
        }
    }

    /**
     * Change Blog Featured.
     */

    public function changeFeatured($id)
    {
            $artists = Artist::findOrFail($id);
            if ($artists->isFeatured == 1) {
                $artists->isFeatured = 2;
                $message = translate('Featured Inactive');
            } else {
                $artists->isFeatured = 1;
                $message = translate('Featured Active');
            }
            if ($artists->update()) {
                return back()->with('success', $message);
            }
    }
}
