<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Location;
use App\Models\ArtGalleryPaymentInfo;
use App\Models\Artwork;
use App\Models\Order;
use App\Models\Transaction;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class ArtGalleryController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $page_title = translate('Art Gallery List');

        $art_galleries = User::where('role', 2)
            ->when($request->search, fn($q) => $q->where(function ($query) use ($request) {
                $search = $request->search;
                $query->where(DB::raw("concat(fname, ' ', lname)"), 'LIKE', "%{$search}%")
                      ->orWhere('email', 'LIKE', "%{$search}%")
                      ->orWhere('custom_id', $search);
            }))
            ->latest()
            ->paginate(12);

        $data = [
            'total_art_galleries' => User::where('role', 2)->count(),
            'total_withdraw'  => Transaction::where('type', 4)->sum('total_amount'),
            'total_received'  => Transaction::where('type', 4)->where('status',2)->sum('total_amount'),
            'total_pending'   => Transaction::where('type', 4)->where('status',1)->sum('total_amount'),
        ];

        return view('backend.art_gallery.index', compact('page_title', 'art_galleries', 'data'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('backend.art_gallery.create', [
            'page_title' => translate('Create Art Gallery'),
            'countries' => Location::whereNull('country_id')->whereNull('state_id')->get(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->validateArtGallery($request);

        $art_gallery = new User($request->except('password', 'image'));
        $art_gallery->password = Hash::make($request->password);
        $art_gallery->role = 2;

        $art_gallery->custom_id = generate_custom_id(2);

        if ($image = $request->file('image')) {
            $art_gallery->image = uploadImage($image);
        }

        if($art_gallery->save()){
            $this->savePaymentInfo($art_gallery->id, $request);
        }

        return redirect()->route('dashboard.art-gallery.index')->with('success', translate('Art Gallery saved successfully'));
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $page_title = translate('Art Gallery Profile');
        $artGallerySingle = User::findOrFail($id);
        $art_gallery_payment = ArtGalleryPaymentInfo::where('user_id', $id)->get();
        $countries = Location::whereNull('country_id')->whereNull('state_id')->get();
        $all_artworks = Artwork::where('author_id', $artGallerySingle->id)->pluck('id');
        $art_gallery_orders = Order::whereIn('artwork_id', $all_artworks)->latest()->paginate(10);
        $artworks = Artwork::where('author_id', $artGallerySingle->id)->latest()->limit(10)->get();

        $data['balance'] = $artGallerySingle->balance;
        $data['withdraw'] = $artGallerySingle->withdraws?->sum('total_amount');
        $data['artwork'] = $artGallerySingle->artworks?->count();
        $data['live_artwork'] = $artGallerySingle->artworks?->where('status',1)->count();

        return view('backend.art_gallery.profile', compact('page_title', 'artGallerySingle', 'countries', 'art_gallery_payment', 'data', 'art_gallery_orders', 'artworks'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $page_title = translate('Edit Art Gallery');
        $artGallerySingle = User::findOrFail($id);
        $art_gallery_payment = ArtGalleryPaymentInfo::where('user_id', $id)->get();
        $countries = Location::whereNull('country_id')->whereNull('state_id')->get();

        return view('backend.art_gallery.edit', compact('page_title', 'artGallerySingle', 'countries', 'art_gallery_payment'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $art_gallery = User::findOrFail($id);
        $this->validateArtGallery($request, $art_gallery->id);

        if ($image = $request->file('image')) {
            if ($art_gallery->image && file_exists(file_path().$art_gallery->image)) {
                unlink(file_path().$art_gallery->image);
            }
            $art_gallery->image = uploadImage($image);
        }

        if ($request->filled('password')) {
            $art_gallery->password = Hash::make($request->password);
        }
        if($art_gallery->update()){
            $this->savePaymentInfo($art_gallery->id, $request);
        }

        return redirect()->route('dashboard.art-gallery.index')->with('success', translate('Art Gallery has been updated successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $art_gallery = User::findOrFail($id);
        if ($art_gallery->image && file_exists(file_path().$art_gallery->image)) {
            unlink(file_path().$art_gallery->image);
        }
        $art_gallery->delete();

        return back()->with('success', translate('Art Gallery deleted successfully'));
    }

    public function changeStatus(Request $request)
    {
        $data_id = $request->data_id;
        if ($data_id) {
            $art_gallery = User::findOrFail($data_id);
            if ($art_gallery->status == 1) {
                $art_gallery->status = 2;
                $message = translate('Deactive');
            } else {
                $art_gallery->status = 1;
                $message = translate('Active');
            }
            if ($art_gallery->update()) {
                $response = array('output' => 'success', 'statusId' => $art_gallery->status, 'dataId' => $art_gallery->id, 'message' => $message);
                return response()->json($response);
            }
        }
    }

    // Private Helpers
    private function validateArtGallery(Request $request, $id = null)
    {
        $uniqueEmail = 'unique:users,email' . ($id ? ",$id" : '');
        $uniquePhone = 'unique:users,phone' . ($id ? ",$id" : '');

        $rules = [
            'fname' => 'required|max:255',
            'lname' => 'required|max:255',
            'email' => "required|max:255|$uniqueEmail",
            'phone' => "required|max:255|$uniquePhone",
            'address' => 'required|max:255',
            'country_id' => 'required',
            'state_id' => 'required',
            'city_id' => 'required',
            'zip_code' => 'required|max:255',
            'password' => ($id ? 'nullable' : 'required') . '|confirmed|min:8',
            'image' => 'nullable|image',
            'admin_commission' => 'nullable|max:255',
        ];

        Validator::make($request->all(), $rules)->validate();
    }



    private function savePaymentInfo($user_id, Request $request)
{
    if (!$request->payment_type) return;
    // Delete all previous payment info for this user
    ArtGalleryPaymentInfo::where('user_id', $user_id)->delete();

    foreach ($request->payment_type as $key => $type) {

        $options = match ((int)$type) {
            1 => [
                'bank_name'           => $request->bank_name[$key] ?? null,
                'branch_name'         => $request->branch_name[$key] ?? null,
                'bank_ac_name'        => $request->bank_ac_name[$key] ?? null,
                'bank_ac_number'      => $request->bank_ac_number[$key] ?? null,
                'bank_routing_number' => $request->bank_routing_number[$key] ?? null,
            ],
            2 => [
                'mobile_banking_name'   => $request->mobile_banking_name[$key] ?? null,
                'mobile_banking_number' => $request->mobile_banking_number[$key] ?? null,
            ],
            3 => [
                'paypal_name'          => $request->paypal_name[$key] ?? null,
                'paypal_username'      => $request->paypal_username[$key] ?? null,
                'paypal_email'         => $request->paypal_email[$key] ?? null,
                'paypal_mobile_number' => $request->paypal_mobile_number[$key] ?? null,
            ],
            default => [],
        };

        ArtGalleryPaymentInfo::create([
            'user_id'       => $user_id,
            'payment_type'  => $type,
            'payment_info'  => json_encode($options),
        ]);
    }
}

/**
     * login
     *
     * @param  int $id
     * @return Response
     */
    public function login($id)
    {
        $art_gallery = User::findOrFail(decrypt($id));

        auth()->login($art_gallery, true);

        return redirect()->route('dashboard.index');
    }
}
