<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $title = 'Register';
        return view('auth.register',compact('title'));
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['nullable', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'integer', 'in:1,2'],
            'agreement' => ['required', 'integer'],
            'g-recaptcha-response' => function ($attribute, $value, $fail) {
                $secretKey = get_setting('recaptcha_secret');
                $response = $value;
                $userIP = $_SERVER['REMOTE_ADDR'];
                $url = "https://www.google.com/recaptcha/api/siteverify?secret=$secretKey&response=$response&remoteip=$userIP";
                $response = Http::asForm()->post($url);
                $response = json_decode($response);
                if (!$response->success) {
                    Session::flash('g-recaptcha-response', 'Please Check Recaptcha');
                    $fail($attribute . "Google Recaptcha Failed");
                }
            }
        ]);

        if($request->role == 2) {
            $request->validate([
                'gallery_name' => ['required', 'string', 'max:255'],
            ]);
        }

        $email_verify = ($request->role == 2) ? (get_setting('artgallry_email_verification') == 1 ? null : now()) : (($request->role == 1) ? (get_setting('customer_email_verification') == 1 ? null : now()) : null);

        $user = User::create([
            'custom_id' => generate_custom_id($request->role),
            'fname' => $request->first_name,
            'lname' => $request->last_name,
            'gallery_name' => $request->gallery_name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'email_verified_at' => $email_verify,
        ]);

        event(new Registered($user));

        Auth::login($user);

        if(Auth::user()->role == 1) {
            return redirect()->intended(route('user.dashboard', absolute: false));
        } else if(Auth::user()->role == 2) {
            return redirect()->intended(route('dashboard.index', absolute: false));
        }

        return redirect(route('user.dashboard', absolute: false));
    }

}
